<?php // phpcs:ignore
/**
 * Api class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/includes/api
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.0.94
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'KP_Api' ) ) {
	/**
	 * Api class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/includes/api
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.0.94
	 *
	 * [Dev] Todo, create a reset api settings option in the bottom footer.
	 */
	class KP_Api {

		/**
		 * Construct the api.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 */
		public function __construct() {
			require_once __DIR__ . '/post.class.php';
			require_once __DIR__ . '/get.class.php';
			require_once __DIR__ . '/send.class.php';
		}

		/**
		 * Logs the API return for debugging and auditing purposes
		 *
		 * @param string $code The response code of the API call.
		 * @param string $message The response message of the API call.
		 * @param mixed  $data Additional data to be logged, such as request parameters or response data.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 * @since 0.0.94
		 */
		public function apiLogger( $code, $message, $data = null ) {
			$file = KP_API_LOG_FILE;
			if ( ! empty( $code ) ) {

				$content = array(
					'code'    => $code,
					'message' => $message,
					'data'    => $data,
				);

				$file_message = $this->getCurrentDate() . ': ' . json_encode( $content ) . "\r\n";
				file_put_contents( $file, $file_message, FILE_APPEND | LOCK_EX );
			}
		}

		/**
		 * Get the current date
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 *
		 * @return string The current date in the format "d-m-y H:i:s".
		 */
		public function getCurrentDate() {
			$date = new DateTime( 'now', new DateTimeZone( 'Europe/Amsterdam' ) );

			return $date->format( 'd-m-y H:i:s' );
		}

		/**
		 * Clear various caching plugins caches
		 *
		 * This function clears the Divi and WP Rocket caches, as well as Litespeed cache and preload the cache if it's available.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 */
		public function clearCache() {
			// Clear the Divi cache.
			$theme = wp_get_theme();
			if ( in_array( 'divi-builder/divi-builder.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) || 'Divi' == $theme->name || 'Divi' == $theme->parent_theme ) {
				ET_Core_PageResource::do_remove_static_resources( 'all', 'all', true );
			}

			// Clear WP Rocket cache.
			// Clear cache.
			if ( function_exists( 'rocket_clean_domain' ) ) {
				rocket_clean_domain();
			}

			// Clear minified CSS and JavaScript files.
			if ( function_exists( 'rocket_clean_minify' ) ) {
				rocket_clean_minify();
			}

			// Create preload.
			if ( function_exists( 'run_rocket_bot' ) ) {
				run_rocket_bot();
			}

			// Clear Litespeed caches.
			do_action( 'litespeed_purge_all' );

			return 'Cache cleared';
		}

		/**
		 * Authenticate the request by checking the secret-key and the https protocol
		 *
		 * @param WP_REST_Request $request The request object.
		 * @return bool|WP_Error True if the request is authenticated, WP_Error object if the request is not authenticated.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 */
		public function authenticate($request) {
			if ( ! is_ssl() ) {
				return new WP_Error( 'https_error', 'HTTPS is required for this endpoint', array( 'status' => 403 ) );
			}

			$header    = $request->get_header('secret-key');
			$secret_key = isset( $header ) ? $header : '';

			if ( ! empty( $secret_key ) ) {
				$stored_secret_key = get_option( 'kls_api_key' );
				if ( $secret_key == $stored_secret_key ) {
					// Check the API secret.
					$this->checkApiSecret();

					update_option( 'kls_api_last_connection', $this->getCurrentDate() );

					return true;
				} else {
					$this->apiLogger( 'secret_key_error', 'Invalid secret key', array( 'status' => 401 ) );
					return new WP_Error( 'secret_key_error', 'Invalid secret key', array( 'status' => 401 ) );
				}
			} else {
				return new WP_Error( 'secret_key_error', 'Missing secret key', array( 'status' => 401 ) );
			}
		}

		/**
		 * Generates a new API key if the current one is the default and updates the last change timestamp
		 *
		 * @return void
		 */
		public function checkApiSecret() {
			$default_secret = 'XlidNb7b3ys4smyjVG3LTnnhlpzJJFlIihnjdL5H7LZ8NGbLrleSE1pjJmPod3ZJ';
			if ( get_option( 'kls_api_key' ) == $default_secret ) {
				update_option( 'kls_api_key', $this->generateApiSecret( 64 ) );
				update_option( 'kls_api_last_change', $this->getCurrentDate() );

				echo esc_url( get_site_url() ) . ',' . esc_html( get_option( 'kls_api_key' ) );
			}
		}

		/**
		 * Generates a random string with a specified length and keyspace
		 *
		 * @param int    $length The desired length of the generated string. Default is 64.
		 * @param string $keyspace The desired keyspace to use for the random string generation. Default is '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 *
		 * @return string The generated random string
		 * @throws \RangeException If the length is less than 1.
		 */
		public function generateApiSecret( int $length = 64, string $keyspace = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ' ): string {
			if ( $length < 1 ) {
				throw new \RangeException( 'Length must be a positive integer' );
			}
			$pieces = array();
			$max    = mb_strlen( $keyspace, '8bit' ) - 1;
			for ( $i = 0; $i < $length; ++$i ) {
				$pieces[] = $keyspace[ random_int( 0, $max ) ];
			}
			return implode( '', $pieces );
		}
	}

	new KP_Api();
}
