<?php // phpcs:ignore
/**
 * Cookie class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/includes/cookie
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.0.92
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'KP_Cookie' ) ) {
	/**
	 * Cookie class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/includes/cookie
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.0.92
	 *
	 * [Dev] Todo, create a reset cookie settings option in the bottom footer.
	 */
	class KP_Cookie {

		/**
		 * Construct the cookie.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 */
		public function __construct() {
			 add_action( 'wp_enqueue_scripts', array( $this, 'enqueueScripts' ) );
			add_action( 'wp_head', array( $this, 'colorsStyling' ) );

			if ( get_option( 'kls_legacy_cookie_toggle' ) == 'on' ) {
				add_action( 'wp_footer', array( $this, 'legacyCookie' ) );
			} else {
				add_action( 'wp_footer', array( $this, 'inlineScript' ) );
				add_filter( 'script_loader_tag', array($this, 'addNowprocketTagToScript'), 10, 3 );
			}

			add_shortcode( 'edit_cookies_button', array( $this, 'cookieEditButtonShortcode' ) );
			add_shortcode( 'edit_cookies_link', array( $this, 'cookieEditLinkShortcode' ) );
		}

		/**
		 * Enqueue the scripts.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 */
		public function enqueueScripts() {
			if ( get_option( 'kls_legacy_cookie_toggle' ) == 'on' ) {
				wp_enqueue_script( 'kls_cookie_script_js', KP_ASSETS_PATH . 'build/public/cookie.js', array( 'jquery' ), KP_VERSION, true );
				wp_enqueue_script( 'js-cookie', '/wp-content/plugins/klasse-plugin/public/js/js.cookie.min.js', array( 'jquery' ), KP_VERSION, true ); // Todo, find a way to move this to the builds directory.
				wp_enqueue_style( 'kls_cookie_css', KP_ASSETS_PATH . 'build/public/cookie.css', false, KP_VERSION );
			} else {
				wp_enqueue_style( 'kls_cookienotice_css', KP_ASSETS_PATH . 'build/public/cookieconsent.css', false, KP_VERSION );
				wp_enqueue_script( 'kls-cookienotice-js', KP_ASSETS_PATH . 'build/public/cookieconsent.js', array( 'jquery' ), KP_VERSION, true );
			}
		}

		/**
		 * Add the nowprocket attribute to the script.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 * 
		 * @since 0.1.10
		 */
		public function addNowprocketTagToScript($tag, $handle, $source)
		{
			if ('kls-cookienotice-js' === $handle) {
				$tag = '<script type="text/javascript" src="' . $source . '" id="kls-cookienotice-js" nowprocket></script>';
			}

			return $tag;
		}

		/**
		 * Edit the cookies shortcode button.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 */
		public function cookieEditButtonShortcode() {
			return '<a href="#" data-cc="c-settings" class="theme-button theme-button-secundary">' . esc_html__( 'Change cookie settings', 'klasse-theme' ) . '</a>';
		}

		/**
		 * Edit the cookies shortcode link.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 */
		public function cookieEditLinkShortcode() {
			 return '<a href="#" data-cc="c-settings" class="button-link">' . esc_html__( 'Change cookie settings', 'klasse-theme' ) . '</a>';
		}

		/**
		 * Legacy Cookie script.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function legacyCookie() {            ?>
			<div id="cookie_notice" class="accepted">
				<div class="notice_container">
					<h3><?php echo kls_plugin_translate_option( 'kls_cookie_title', 'User input cookie notice title', 'nl' ); ?></h3>
					<p><?php echo kls_plugin_translate_option( 'kls_cookie_text', 'User input cookie notice text', 'nl' ); ?></p>
					<div class="notice_button_container">
						<a class="notice_button notice_accept" style="color: <?php echo get_option( 'kls_cookie_button_color', '#ffffff' ); ?>; background-color: <?php echo get_option( 'kls_cookie_button_background_color', '#008000' ); ?>;"><?php echo kls_plugin_translate_option( 'kls_cookie_button_text', 'User input cookie notice button text', 'nl' ); ?></a>
						<?php
						if ( 'on' == get_option( 'kls_cookie_link_page' ) ) :
							?>
							<a href="<?php echo get_option('kls_cookie_link'); ?>" class="notice_button notice_readmore">
								<?php echo kls_plugin_translate_option('kls_cookie_link_text', 'User input cookie notice policy text', 'nl'); ?>
							</a>

						<?php else: ?>
							<a href="<?php echo get_permalink(kls_plugin_translate_option('kls_cookie_link', 'User input cookie notice policy link', 'nl')); ?>" class="notice_button notice_readmore">
								<?php echo kls_plugin_translate_option('kls_cookie_link_text', 'User input cookie notice policy text', 'nl'); ?>
							</a>
						<?php endif; ?>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * The inline cookie script.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 *
		 * [Dev] Todo, only show specific settings modals if those specific settings are being used.
		 */
		public function inlineScript() {
			?>
			<script nowprocket>
			window.addEventListener('load', function(){

				// obtain plugin
				var cc = initCookieConsent();

				// run plugin with your configuration
				cc.run({
					current_lang: 'nl',
					autoclear_cookies: true,
					page_scripts: true,
					cookie_expiration: <?php echo get_option( 'kls_cookie_expiration', '180' ); ?>,

					<?php if ( 'on' == get_option( 'kls_cookie_consentlog_toggle' ) ) : ?>
						onFirstAction: function(user_preferences, cookie){
							if( user_preferences.accept_type == 'all') {
								var consent_status = 'accepted';
							} else {
								var consent_status = 'rejected';
							}
							jQuery.ajax({
								method: 'post',
								url: '<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>',
								data: {
									action: 'logConsent',
									security: '<?php echo esc_html( wp_create_nonce( 'kls_ajax_validation' ) ); ?>',
									status: consent_status,
									accepted_categories: user_preferences.accepted_categories,
									rejected_categories: user_preferences.rejected_categories,
								}
							});
						},
					<?php endif; ?>

					gui_options: {
						consent_modal: {
							layout: '<?php echo get_option( 'kls_cookie_bar_styling', 'box' ); ?>',
							position: '<?php echo get_option( 'kls_cookie_bar_location_vertical', 'bottom' ); ?> <?php echo get_option( 'kls_cookie_bar_location_horizontal', 'left' ); ?>',
							transition: '<?php echo get_option( 'kls_cookie_bar_transition', 'zoom' ); ?>',
							swap_buttons: false
						},
						settings_modal: {
							layout: 'box', 
							transition: 'slide'
						}
					},

					languages: {
						'nl': {
							consent_modal: {
								title: "<?php echo esc_html( kls_plugin_translate_option( 'kls_cookie_title', 'User input cookie notice title', 'nl' ) ); ?>",
								description: "<?php echo esc_html( kls_plugin_translate_option( 'kls_cookie_text', 'User input cookie notice text', 'nl' ) ); ?>",
								primary_btn: {
									text: "<?php echo esc_html( kls_plugin_translate_option( 'kls_cookie_button_text', 'User input cookie notice button text', 'nl' ) ); ?>",
									role: 'accept_all'
								},
								secondary_btn: {
									text: "<?php echo esc_html( kls_plugin_translate_option( 'kls_cookie_link_text', 'User input cookie notice policy text', 'nl' ) ); ?>",
									role: 'settings'
								}
							},
							settings_modal: {
								title: "<?php echo get_option( 'kls_cookie_popup_title' ); ?>",
								save_settings_btn: "<?php echo get_option( 'kls_cookie_popup_save_settings_text' ); ?>",
								accept_all_btn: "<?php echo get_option( 'kls_cookie_popup_accept_all_text' ); ?>",
								reject_all_btn: "<?php echo get_option( 'kls_cookie_popup_reject_all_text' ); ?>",
								close_btn_label: "Close",
								blocks: [
									{
										title: "<?php echo get_option( 'kls_cookie_popup_intro_title' ); ?>",
										description: '<?php echo get_option( 'kls_cookie_popup_intro_text' ); ?>',
									}, {
										title: "<?php echo get_option( 'kls_cookie_popup_necessary_title' ); ?>",
										description: "<?php echo get_option( 'kls_cookie_popup_necessary_text' ); ?>",
										toggle: {
											value: 'necessary',
											enabled: true,
											readonly: true,
										},
									}, 
									<?php
									if ( $this->inArrayR( 'analytics', get_option( 'kls_script_selector' ) ) || ! empty( get_option( 'kls_scripts_ua' ) ) ) :
										?>
										{
											title: "<?php echo get_option( 'kls_cookie_popup_analytics_title' ); ?>",
											description: "<?php echo get_option( 'kls_cookie_popup_analytics_text' ); ?>",
											toggle: {
												value: 'analytics',
												enabled: <?php echo get_option( 'kls_cookie_analytics_enabled', 'off' ) == 'on' ? 'true' : 'false'; ?>,
												readonly: false,
											},
										}, 
										<?php
									endif;
									if ( $this->inArrayR( 'marketing', get_option( 'kls_script_selector' ) ) || ! empty( get_option( 'kls_scripts_gtm' ) ) || ! empty( get_option( 'kls_scripts_fbpixel' ) ) ) :
										?>
										{
											title: "<?php echo get_option( 'kls_cookie_popup_marketing_title' ); ?>",
											description: "<?php echo get_option( 'kls_cookie_popup_marketing_text' ); ?>",
											toggle: {
												value: 'marketing',
												enabled: <?php echo get_option( 'kls_cookie_marketing_enabled', 'off' ) == 'on' ? 'true' : 'false'; ?>,
												readonly: false,
											},
										}, 
										<?php
									endif;
									if ( get_option( 'kls_cookie_popup_more_info_toggle' ) == 'on' ) :
										?>
										{
											title: "<?php echo get_option( 'kls_cookie_popup_more_info_title' ); ?>",
											description: '<?php echo get_option( 'kls_cookie_popup_more_info_text' ); ?>',
										},
									<?php endif; ?>
								]
							}
						}
					}
				});
			});
			</script>
			<?php
		}

		/**
		 * Add the styling for the cookie notice.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function colorsStyling() {
			?>
			<style>
				:root {
					--cc-bg: <?php echo get_option( 'kls_cookie_background_color', '#fff' ); ?>;
					--cc-text: <?php echo get_option( 'kls_cookie_body_text_color', '#222' ); ?>;
					--cc-border-radius: .35em;
					<?php
					if ( 'theme' == get_option( 'kls_cookie_button_styling' ) ) :
						if ( wp_get_theme() == 'Klasse theme' || wp_get_theme() == 'Klasse Theme' || wp_get_theme() == 'Klasse child theme' || wp_get_theme() == 'Klasse Child Theme' ) :
							?>
							--cc-btn-primary-text: <?php echo get_theme_mod( 'button_primary_color', '#fff' ); ?>;
							--cc-btn-primary-bg: <?php echo get_theme_mod( 'button_primary_background_color', '#008000' ); ?>;
							--cc-btn-primary-hover-text: <?php echo get_theme_mod( 'button_primary_hover_color', '#008000' ); ?>;
							--cc-btn-primary-hover-bg: <?php echo get_theme_mod( 'button_primary_hover_background_color', '#fff' ); ?>;
							--cc-btn-secondary-text: <?php echo get_theme_mod( 'button_secundary_color', '#222' ); ?>;
							--cc-btn-secondary-bg: <?php echo get_theme_mod( 'button_secundary_background_color', '#eee' ); ?>;
							--cc-btn-secondary-hover-text: <?php echo get_theme_mod( 'button_secundary_hover_color', '#fff' ); ?>;
							--cc-btn-secondary-hover-bg: <?php echo get_theme_mod( 'button_secundary_hover_background_color', '#222' ); ?>;
							--cc-btn-border-radius: <?php echo get_theme_mod( 'button_primary_border_radius', '4px' ); ?>;
							<?php
						endif;
					else :
						?>
						--cc-btn-primary-text: <?php echo get_option( 'kls_cookie_primary_button_color', '#fff' ); ?>;
						--cc-btn-primary-bg: <?php echo get_option( 'kls_cookie_primary_button_background_color', '#008000' ); ?>;
						--cc-btn-primary-hover-text: <?php echo get_option( 'kls_cookie_primary_button_hover_color', '#008000' ); ?>;
						--cc-btn-primary-hover-bg: <?php echo get_option( 'kls_cookie_primary_button_hover_background_color', '#fff' ); ?>;
						--cc-btn-secondary-text: <?php echo get_option( 'kls_cookie_secundary_button_color', '#222' ); ?>;
						--cc-btn-secondary-bg: <?php echo get_option( 'kls_cookie_secundary_button_background_color', '#eee' ); ?>;
						--cc-btn-secondary-hover-text: <?php echo get_option( 'kls_cookie_secundary_button_hover_color', '#fff' ); ?>;
						--cc-btn-secondary-hover-bg: <?php echo get_option( 'kls_cookie_secundary_button_hover_background_color', '#222' ); ?>;
						--cc-btn-border-radius: <?php echo get_option( 'kls_cookie_button_border_radius', '4px' ); ?>;
						<?php
					endif;
					?>
					--cc-toggle-bg-off: #919ea6;
					--cc-toggle-bg-on: var(--cc-btn-primary-bg);
					--cc-toggle-bg-readonly: #d5dee2;
					--cc-toggle-knob-bg: #fff;
					--cc-toggle-knob-icon-color: #ecf2fa;
					--cc-block-text: var(--cc-text);
					--cc-cookie-category-block-bg: <?php echo get_option( 'kls_cookie_table_background_color', '#f0f4f7' ); ?>;
					--cc-cookie-category-block-bg-hover: <?php echo get_option( 'kls_cookie_table_background_color', '#f0f4f7' ); ?>;
					--cc-section-border: #f1f3f5;
					--cc-cookie-table-border: #e9edf2;
					--cc-overlay-bg: rgba(4, 6, 8, .85);
					--cc-webkit-scrollbar-bg: #cfd5db;
					--cc-webkit-scrollbar-bg-hover: #9199a0
				}
			</style>
			<?php
		}

		public function inArrayR( $needle, $haystack, $strict = false ) {
			foreach ( $haystack as $item ) {
				if ( ( $strict ? $item === $needle : $item == $needle ) || ( is_array( $item ) && $this->inArrayR( $needle, $item, $strict ) ) ) {
					return true;
				}
			}

			return false;
		}
	}

	new KP_Cookie();
}
