<?php // phpcs:ignore
/**
 * Scripts class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/includes/scripts
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.0.93
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
	exit;
}

if (!class_exists('KP_Scripts')) {
	/**
	 * Scripts class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/includes/scripts
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.0.93
	 */
	class KP_Scripts
	{

		/**
		 * Construct the scripts.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function __construct()
		{
			add_action('wp_head', array($this, 'addCookieScriptCode'), 1);

			if ('on' == get_option('kls_legacy_scripts_toggle')) {
				add_action('wp_head', array($this, 'legacyAddScriptsHead'));
				add_action('wp_body_open', array($this, 'legacyAddScriptsBody'));
				add_action('wp_footer', array($this, 'legacyAddScriptsFooter'));
			} else {

				add_action('wp_head', array($this, 'addCustomScriptsHead'));
				add_action('wp_body_open', array($this, 'addCustomScriptsBody'));
				add_action('wp_footer', array($this, 'addCustomScriptsFooter'));

				add_shortcode('custom_script', array($this, 'addCustomScriptsShortcode'));
			}

			add_action('wp_head', array($this, 'addSearchConsole'));
			add_action('wp_head', array($this, 'addGoogleAnalytics'));
			add_action('wp_head', array($this, 'addGoogleTagManagerHead'));
			add_action('wp_body_open', array($this, 'addGoogleTagManagerBody'));
			add_action('wp_head', array($this, 'addFacebookPixel'));
			add_action('wp_head', array($this, 'addLandbot'));
		}

		/**
		 * Add the custom scripts to the head.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function addCustomScriptsHead()
		{
			$options = get_option('kls_script_selector');

			if (isset($options) && !empty($options)) {
				foreach ($options as $value) {
					if (isset($value['toggle']) && isset($value['location']) && 'head' == $value['location'] && 'on' == $value['toggle']) {
						$this->doCustomScript($value);
					}
				}
			}
		}

		/**
		 * Add the custom scripts to the body.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function addCustomScriptsBody()
		{
			$options = get_option('kls_script_selector');

			if (isset($options) && !empty($options)) {
				foreach ($options as $value) {
					if (isset($value['toggle']) && isset($value['location']) && 'body' == $value['location'] && 'on' == $value['toggle']) {
						$this->doCustomScript($value);
					}
				}
			}
		}

		/**
		 * Add the custom scripts to the head.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function addCustomScriptsFooter()
		{
			$options = get_option('kls_script_selector');

			if (isset($options) && !empty($options)) {
				foreach ($options as $value) {
					if (isset($value['toggle']) && isset($value['location']) && 'footer' == $value['location'] && 'on' == $value['toggle']) {
						$this->doCustomScript($value);
					}
				}
			}
		}

		/**
		 * Add a custom scripts as a script.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function addCustomScriptsShortcode($atts)
		{
			extract(
				shortcode_atts(
					array(
						'name' => 1,
					),
					$atts
				)
			);

			$options = get_option('kls_script_selector');

			if (isset($options) && !empty($options)) {
				foreach ($options as $value) {
					if (isset($value['toggle']) && isset($value['location']) && 'shortcode' == $value['location'] && 'on' == $value['toggle']) {
						ob_start();

						$this->doCustomScript($value);

						return ob_get_clean();
					}
				}
			}
		}

		/**
		 * Process the custom script.
		 *
		 * @param array $value An array comming from the option 'kls_script_selector'.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.93
		 */
		public function doCustomScript($value)
		{
			if (isset($value['cookie_category']) && !'none' == $value['cookie_category']) {
				$value['script'] = str_replace('<script', '<script type="text/plain" data-cookiecategory="' . $value['cookie_category'] . '"', $value['script']);
			}

			echo $value['script'];
		}

		/**
		 * Include search console code
		 *
		 * @since 0.0.86
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addSearchConsole()
		{
			if (get_option('kls_scripts_searchconsole')) {
				$option = get_option('kls_scripts_searchconsole');
				$codes = rtrim($option, ',');
				$codes = explode(',', $codes);
				$codes = array_unique($codes);

				foreach ($codes as $code):
					?>
					<meta name="google-site-verification" content="<?php echo esc_html($code); ?>" />
					<?php
				endforeach;
			}
		}

		/**
		 * Include analytics code
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addGoogleAnalytics()
		{
			if (get_option('kls_scripts_ua')) {
				$option = get_option('kls_scripts_ua');
				$codes = rtrim($option, ',');
				$codes = explode(',', $codes);
				$codes = array_unique($codes);

				$cookie_code = 'type="text/plain" data-cookiecategory="analytics"';

				foreach ($codes as $code):
					?>
					<!-- Global site tag (gtag.js) - Google Analytics -->
					<script async src="https://www.googletagmanager.com/gtag/js?id=<?php echo esc_html($code); ?>" <?php echo (!'on' == get_option('kls_legacy_cookie_toggle') && 'on' == get_option('kls_cookie_toggle')) ? $cookie_code : ''; ?>></script>
					<script <?php echo (!'on' == get_option('kls_legacy_cookie_toggle') && 'on' == get_option('kls_cookie_toggle')) ? $cookie_code : ''; ?>>
						window.dataLayer = window.dataLayer || [];
						function gtag() { dataLayer.push(arguments); }
						gtag('js', new Date());

						gtag('config', '<?php echo esc_html($code); ?>');
					</script>
					<?php
				endforeach;
			}
		}

		/**
		 * Include Google tag manager head
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addGoogleTagManagerHead()
		{
			if (get_option('kls_scripts_gtm')) {
				$option = get_option('kls_scripts_gtm');
				$codes = rtrim($option, ',');
				$codes = explode(',', $codes);
				$codes = array_unique($codes);

				foreach ($codes as $code):
					?>
					<!-- Google Tag Manager -->
					<script <?php echo (!'on' == get_option('kls_legacy_cookie_toggle') && 'on' == get_option('kls_cookie_toggle')) ? 'type="text/plain" data-cookiecategory="marketing"' : ''; ?>>
						<?php if (get_option('kls_scripts_gtm_optimization') == 'on'): ?>
							window.addEventListener('mouseover', initGTM<?php echo esc_html($code); ?>, { once: true });
							window.addEventListener('touchstart', initGTM<?php echo esc_html($code); ?>, { once: true });
							window.addEventListener('scroll', initGTM<?php echo esc_html($code); ?>, { once: true });
							function initGTM<?php echo esc_html($code); ?>() {
							<?php endif; ?>
								(function (w, d, s, l, i) {
									w[l] = w[l] || []; w[l].push({
										'gtm.start':
											new Date().getTime(), event: 'gtm.js'
									}); var f = d.getElementsByTagName(s)[0],
										j = d.createElement(s), dl = l != 'dataLayer' ? '&l=' + l : ''; j.async = true; j.src =
											'https://www.googletagmanager.com/gtm.js?id=' + i + dl; f.parentNode.insertBefore(j, f);
								})(window, document, 'script', 'dataLayer', '<?php echo esc_html($code); ?>');
							<?php if (get_option('kls_scripts_gtm_optimization') == 'on'): ?>
							};
						<?php endif; ?>
					</script>
					<!-- End Google Tag Manager -->
					<?php
				endforeach;
			}
		}

		/**
		 * Include Google tag manager body
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addGoogleTagManagerBody()
		{
			if (get_option('kls_scripts_gtm')) {
				$option = get_option('kls_scripts_gtm');
				$codes = rtrim($option, ',');
				$codes = explode(',', $codes);
				$codes = array_unique($codes);

				foreach ($codes as $code):
					?>
					<!-- Google Tag Manager (noscript) -->
					<noscript><iframe src="https://www.googletagmanager.com/ns.html?id=<?php echo esc_html($code); ?>" height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>
					<!-- End Google Tag Manager (noscript) -->
					<?php
				endforeach;
			}
		}

		/**
		 * Include Facebook pixel
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addFacebookPixel()
		{
			if (get_option('kls_scripts_fbpixel')) {
				$option = get_option('kls_scripts_fbpixel');
				$codes = rtrim($option, ',');
				$codes = explode(',', $codes);
				$codes = array_unique($codes);

				foreach ($codes as $code):
					?>
					<!-- Facebook Pixel Code -->
					<script <?php echo (!'on' == get_option('kls_legacy_cookie_toggle') && 'on' == get_option('kls_cookie_toggle')) ? 'type="text/plain" data-cookiecategory="marketing"' : ''; ?>>
						<?php if (get_option('kls_scripts_fbpixel_optimization') == 'on'): ?>
							window.addEventListener('mouseover', initFB<?php echo esc_html($code); ?>, { once: true });
							window.addEventListener('touchstart', initFB<?php echo esc_html($code); ?>, { once: true });
							window.addEventListener('scroll', initFB<?php echo esc_html($code); ?>, { once: true });
							function initFB<?php echo esc_html($code); ?>() {
							<?php endif; ?>
							!function (f, b, e, v, n, t, s) {
								if (f.fbq) return; n = f.fbq = function () {
									n.callMethod ?
										n.callMethod.apply(n, arguments) : n.queue.push(arguments)
								};
								if (!f._fbq) f._fbq = n; n.push = n; n.loaded = !0; n.version = '2.0';
								n.queue = []; t = b.createElement(e); t.async = !0;
								t.src = v; s = b.getElementsByTagName(e)[0];
								s.parentNode.insertBefore(t, s)
							}(window, document, 'script',
								'https://connect.facebook.net/en_US/fbevents.js');
							fbq('init', '<?php echo esc_html($code); ?>');
							fbq('track', 'PageView');
							<?php if (get_option('kls_scripts_fbpixel_optimization') == 'on'): ?>
							}
						<?php endif; ?>
					</script>
					<noscript>
						<img height="1" width="1" style="display:none" src="https://www.facebook.com/tr?id=<?php echo esc_html($code); ?>&ev=PageView&noscript=1" />
					</noscript>
					<!-- End Facebook Pixel Code -->
					<?php
				endforeach;
			}
		}

		/**
		 * Include Landbot Chatbot
		 *
		 * @since 0.0.82
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function addLandbot()
		{
			if (get_option('kls_scripts_chatbot')) {
				?>
				<!-- Landbot Chatbot Code -->
				<script>
					window.addEventListener('mouseover', initLandbot, { once: true });
					window.addEventListener('touchstart', initLandbot, { once: true });
					window.addEventListener('scroll', initLandbot, { once: true });
					var myLandbot;
					function initLandbot() {
						if (!myLandbot) {
							var s = document.createElement('script'); s.type = 'text/javascript'; s.async = true;
							s.addEventListener('load', function () {
								var myLandbot = new Landbot.Livechat({
									configUrl: '<?php echo esc_html(get_option('kls_scripts_chatbot')); ?>',
								});
							});
							s.src = 'https://cdn.landbot.io/landbot-3/landbot-3.0.0.js';
							var x = document.getElementsByTagName('script')[0];
							x.parentNode.insertBefore(s, x);
						}
					}
				</script>
				<!-- End Landbot Chatbot Code -->
				<?php
			}
		}

		/**
		 * Add cookie-script.com script.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 * 
		 * @since 0.1.19
		 */
		public function addCookieScriptCode()
		{
			if (get_option('kls_scripts_cookie_script')) {
				?>	<script type="text/javascript" charset="UTF-8" src="<?php echo esc_html(get_option('kls_scripts_cookie_script')); ?>"></script><?php
			}
		}


		// ------------- Under this line are the legacy functions ------------- //

		/**
		 * Include header scripts
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function legacyAddScriptsHead()
		{
			if (get_option('kls_scripts_head')) {
				echo get_option('kls_scripts_head');
			}
		}

		/**
		 * Include body scripts
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function legacyAddScriptsBody()
		{
			if (get_option('kls_scripts_body')) {
				echo get_option('kls_scripts_body');
			}
		}

		/**
		 * Include header scripts
		 *
		 * @since 0.0.05
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 */
		public function legacyAddScriptsFooter()
		{
			if (get_option('kls_scripts_footer')) {
				echo get_option('kls_scripts_footer');
			}
		}
	}

	new KP_Scripts();
}
