<?php // phpcs:ignore
/**
 * Link integration class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/shortcodes/linkintegration
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.0.95
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'KP_LinkIntegration' ) ) {
	/**
	 * Link integration class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/shortcodes/linkintegration
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.0.95
	 */
	class KP_LinkIntegration {

		/**
		 * Construct the link integration.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.95
		 */
		public function __construct() {
			add_action( 'init', array( $this, 'addCategoriesToPages' ) );

			add_shortcode( 'link_integratie', array( $this, 'linkIntegrationShortcode' ) );
		}

		/**
		 * Adds the 'category' taxonomy to the 'page' post type.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.95
		 */
		public function addCategoriesToPages() {
			register_taxonomy_for_object_type( 'category', 'page' );
		}

		/**
		 *
		 * Generates a list of links to pages in the specified category and outputs them as a shortcode.
		 *
		 * @param array $atts An array of shortcode attributes.
		 * - category (string): The name of the category to get pages from. Required.
		 * - amount (int): The number of links to display. Optional. Default is 3.
		 * - seperator (string): The string to use as a separator between links. Optional. Default is '|'.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.95
		 *
		 * @return string The HTML output for the shortcode.
		 */
		public function linkIntegrationShortcode( $atts ) {
			global $post;

			// Get the shortcode attributes.
			$category  = isset( $atts['category'] ) ? $atts['category'] : '';
			$amount    = isset( $atts['amount'] ) ? $atts['amount'] : 3;
			$seperator = isset( $atts['seperator'] ) ? $atts['seperator'] : '|';

			$output = '';
			if ( $category ) {
				// Construct the links array.
				$links = $this->generateLinksArray( $category );

				// Generate the output.
				if ( $links ) {
					$j                = 0;
					$current_page_key = array_search( get_the_ID(), array_column( $links, 'id' ) );

					$output  = '<div class="seo-pages-links">';
					$output .= '<p>' . __( 'Check out the following links:', KP_TXT_DOMAIN ) . '</p><ul>';

					for ( $i = 1; $i <= $amount; $i++ ) {
						$page_key      = $current_page_key + $i; // Get the page key including the amount from the for loop.
						$get_first_key = isset( $links[ $page_key ] ) ? 0 : $j++; // Generate a new key if there isn't a current key.

						$link           = isset( $links[ $page_key ] ) ? $links[ $page_key ] : $links[ $get_first_key ];
						$seperator_html = $i < $amount ? ' ' . $seperator . ' ' : ''; // Only output an seperator if it isn't the last

						$output .= '<a href="' . $link['url'] . '">' . $link['title'] . '</a>' . $seperator_html;
					}

					$output .= '</ul></div>';
				}
			}
			return $output;
		}

		/**
		 * Generates an array of links for pages in the specified category.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.95
		 *
		 * @param string $category The name of the category to get pages from.
		 * @return array An array of links, where each link is an associative array with the following keys:
		 * - id (int): The post ID of the page.
		 * - url (string): The URL of the page.
		 * - title (string): The title of the page.
		 */
		public function generateLinksArray( $category ) {
			// Construct the WP_Query to show pages in the category.
			$args            = array(
				'category_name'  => $category,
				'post_type'      => 'page',
				'posts_per_page' => -1,
				'orderby'        => 'title',
				'order'          => 'ASC',
			);
			$seo_pages_query = new WP_Query( $args );

			// Construct the links query.
			$i     = 0;
			$links = array();

			while ( $seo_pages_query->have_posts() ) {
				$seo_pages_query->the_post();

				$post_id    = get_the_id();
				$post_url   = get_permalink();
				$post_title = get_the_title();

				$links[ $i ]['id']    = $post_id;
				$links[ $i ]['url']   = $post_url;
				$links[ $i ]['title'] = $post_title;

				$i++;
			}

			wp_reset_postdata();

			return $links;
		}
	}

	new KP_LinkIntegration();
}
