<?php
/**
 * Enqueue font heading scripts
 *
 * @since 0.0.21
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * [ToDo] Dev, add preconnect to the fonts api
 */
if ( get_theme_mod( 'google_fonts_toggle' ) == 1 ) {
	function kls_fonts_styles() {
		$headings_font = esc_html( get_theme_mod( 'kls_headings_font' ) );
		$body_font     = esc_html( get_theme_mod( 'kls_body_font' ) );

		// Headings font
		if ( $headings_font ) {
			wp_enqueue_style( 'kls-headings-fonts', '//fonts.googleapis.com/css?family=' . $headings_font . '&display=swap' );
		} else {
			wp_enqueue_style( 'kls-standard-heading-font', '//fonts.googleapis.com/css?family=Open+Sans%3A300%2C300italic%2Cregular%2Citalic%2C600%2C600italic%2C700%2C700italic%2C800%2C800italic&display=swap' );
		}

		// Body font
		if ( $body_font ) {
			wp_enqueue_style( 'kls-body-fonts', '//fonts.googleapis.com/css?family=' . $body_font . '&display=swap' );
		} else {
			wp_enqueue_style( 'kls-standard-body-font', '//fonts.googleapis.com/css?family=Open+Sans%3A300%2C300italic%2Cregular%2Citalic%2C600%2C600italic%2C700%2C700italic%2C800%2C800italic&display=swap' );
		}
	}
	add_action( 'wp_enqueue_scripts', 'kls_fonts_styles' );
}

/**
 * Require all Google fonts
 *
 * @since 0.0.21
 * @author Joeri van der Stek <joeri@klassebv.nl>
 */
require_once __DIR__ . '/googlefonts.php';

/**
 * Insert the heading font to the head
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.45
 */
function kls_insert_heading_font() {
	echo kls_create_custom_font( get_theme_mod( 'kls_custom_fonts_heading' ) );
}
add_action( 'wp_head', 'kls_insert_heading_font' );

/**
 * Insert the body font to the head
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.45
 */
function kls_insert_body_font() {
	echo kls_create_custom_font( get_theme_mod( 'kls_custom_fonts_body' ) );
}
add_action( 'wp_head', 'kls_insert_body_font' );

/**
 * Generate font styling
 *
 * @param string $font_string
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.45
 *
 * @return string
 *
 * [ToDo] Dev, clean up
 * [ToDo] Dev, create preload
 */
function kls_create_custom_font( $font_string ) {
	// Remove last comma
	$fonts = rtrim( $font_string, ',' );
	// Explode into an array
	$fonts = explode( ',', $fonts );

	// Combine different formats in 1 array
	$count = 0;
	foreach ( $fonts as $font ) :
		$font_output = preg_replace( '/^.*\/\s*/', '', $font ); // Get only the output with the file extension.
		$font_weight = strtolower( get_string_between( $font_output, '-', '.' ) ); // Get the font weight and style.
		$font_output = strstr( $font_output, '.', true ); // Remove the file extension.
		$font_name   = substr( $font_output, 0, strpos( $font_output, '-' ) ); // Get the global font name

		// Generate the font format
		$font_format = preg_replace( '/^.*\.\s*/', '', $font );

		// Create the array
		$fonts_combined[ $font_output ][ $count ]['url']         = $font;
		$fonts_combined[ $font_output ][ $count ]['name']        = $font_output;
		$fonts_combined[ $font_output ][ $count ]['format']      = $font_format;
		$fonts_combined[ $font_output ][ $count ]['global_name'] = $font_name;
		$fonts_combined[ $font_output ][ $count ]['weight']      = $font_weight;

		$count++;
	endforeach;

	// Sort the array by font format
	$standard_sort = array( 'eot', 'woff2', 'woff', 'ttf', 'otf', 'svg' );
	foreach ( $fonts_combined as $font_combined ) :
		// Sort the array by the standard sort array.
		usort(
			$font_combined,
			function ( $a, $b ) use ( $standard_sort ) {
				$pos_a = array_search( $a['format'], $standard_sort );
				$pos_b = array_search( $b['format'], $standard_sort );
				return $pos_a - $pos_b;
			}
		);

		$fonts_sorted[] = $font_combined;
	endforeach;

	// Generate CSS
	echo '<style>'; // Start style
	foreach ( $fonts_sorted as $font ) :
		$font_name   = $font[0]['global_name'];
		$font_weight = kls_find_font_weight( $font[0]['weight'] );
		$font_style  = ( strstr( $font[0]['weight'], 'italic' ) ? 'italic' : 'normal' );

		// Generate the font face styling
		echo "@font-face {
            font-family: '" . $font_name . "';
            ";

			// Generate all the font sources
			$count  = 0;
			$length = count( $font );

		foreach ( $font as $extra_font ) :
			$font_format = kls_find_font_format( $extra_font['format'] );

			$count++;

			// Add the first font without the format
			if ( $count == 1 && $font_format == 'embedded-opentype' ) :
				echo "src: url('" . $extra_font['url'] . "');";
				endif;

			// Add the specific font prefixes
			if ( $font_format == 'embedded-opentype' ) : // Add the EOT iefix to the url
				$extra_font['url'] = $extra_font['url'] . '?#iefix';
				elseif ( $font_format == 'svg' ) : // Add the SVG #name to the url
					$extra_font['url'] = $extra_font['url'] . '#' . $extra_font['name'];
				endif;

				// Add all the other fonts
				if ( $count == 1 && $count == $length ) : // If the count is 1 and it's the last string in the array
					echo "src: url('" . $extra_font['url'] . "') format('" . $font_format . "');";
				elseif ( $count == 1 && $count !== $length ) : // If the count is 1 and it isn't the last string in the array
					echo "src: url('" . $extra_font['url'] . "') format('" . $font_format . "'),";
				elseif ( $count < $length ) : // If not last item
					echo "url('" . $extra_font['url'] . "') format('" . $font_format . "'),";
				elseif ( $count == $length ) : // If last item
					echo "url('" . $extra_font['url'] . "') format('" . $font_format . "');";
				endif;
			endforeach;

		echo '
            font-weight: ' . $font_weight . ';
            font-style: ' . $font_style . ';
            font-display: swap;
        }';
	endforeach;
	echo '</style>'; // End style
}

/**
 * Return the font weight
 *
 * @param string $font_weight
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.45
 *
 * @return string
 */
function kls_find_font_weight( $font_weight ) {
	if ( strstr( $font_weight, 'black' ) ) :
		return '900';
	elseif ( strstr( $font_weight, 'extrabold' ) ) :
		return '800';
	elseif ( strstr( $font_weight, 'semibold' ) ) :
		return '600';
	elseif ( strstr( $font_weight, 'bold' ) ) :
		return '700';
	elseif ( strstr( $font_weight, 'medium' ) ) :
		return '500';
	elseif ( strstr( $font_weight, 'normal' ) || strstr( $font_weight, 'regular' ) ) :
		return '400';
	elseif ( strstr( $font_weight, 'extralight' ) ) :
		return '200';
	elseif ( strstr( $font_weight, 'light' ) ) :
		return '300';
	elseif ( strstr( $font_weight, 'thin' ) ) :
		return '100';
	else :
		return 'normal';
	endif;
}

/**
 * Get the specific font format
 *
 * @param string $font_format
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.45
 *
 * @return string
 */
function kls_find_font_format( $font_format ) {
	if ( strstr( $font_format, 'eot' ) ) :
		return 'embedded-opentype';
	elseif ( strstr( $font_format, 'woff2' ) ) :
		return 'woff2';
	elseif ( strstr( $font_format, 'woff' ) ) :
		return 'woff';
	elseif ( strstr( $font_format, 'svg' ) ) :
		return 'svg';
	elseif ( strstr( $font_format, 'ttf' ) ) :
		return 'truetype';
	elseif ( strstr( $font_format, 'otf' ) ) :
		return 'opentype';
	endif;
}
